<?php

namespace RSSAutopilot;

require_once(RSSAP_PLUGIN_DIR.'/classes/validate-form.php');

/**
 * Settings controller
 */
class Settings {
    /**
     * Access to bootstrap object
     */
    private $bootstrap = null;

    public function __construct($bootstrap=null)
    {
        $this->bootstrap = $bootstrap;

        if ( is_admin() ) {
            $screen = get_current_screen();
            $screen->add_help_tab(array(
                'id' => 'rssap-feeds-help',
                'title' => __('Settings', 'rss-autopilot'),
                'content' =>
                    '<p>' . __('On this page you can configure RSS AutoPilot plugin. At the moment there is only one Yandex Translator API key available.', 'rss-autopilot') . '</p>' .
                    '<p>' . __('For more details check <a href="' . rssap_plugin_url('RSSAutopilot-UserManual.pdf') . '">User Manual</a>', 'rss-autopilot') . '</p>'
            ));
        }
    }

    /**
     * Settings page
     * @return array
     */
    public function index()
    {
        $request = $this->bootstrap->getRequest();

        if ($request->getRequest('success')) {
            add_action( 'admin_notices', array($this, 'displayNotice') );
        }

        wp_enqueue_script(
            'rss-autopilot-settings',
            rssap_plugin_url('admin/js/settings.js'),
            array( 'jquery' ),
            RSSAP_VERSION,
            'all'
        );

        return array(
            'yandex_api_key' => get_option( '_rssap_yandex_translator_api_key' )
        );
    }

    /**
     * Save settings
     * @return array
     */
    function save()
    {
        $request = $this->bootstrap->getRequest();
        $data = $request->getPost();

        if (!empty($data) && $request->isAjaxRequest()) {
            check_admin_referer( 'rssap-save-rss-autopilot' );

            $form = $this->getSettingsFormValidator($data);

            if (!$form->isValid()) {
                $this->sendAjaxRespone(false, $form->validate());
            } else {
                update_option( '_rssap_yandex_translator_api_key', $data['yandex_translator_api_key'] );

                $redirectUrl = $this->bootstrap->menuUrl('rssap-settings').'&success=1';
                $this->sendAjaxRespone(true, array(), array(), $redirectUrl);
            }
        }

        return array();
    }

    /**
     * Send AJAX response of a specified format
     * @param $status
     * @param array $errors
     * @param string $redirectUrl
     */
    function sendAjaxRespone($status, $errors=array(), $data=array(), $redirectUrl = '')
    {
        $response = array (
            'status' => $status
        );

        if ($errors && count($errors)) {
            $response['errors'] = $errors;
        }

        if ($data && count($data)) {
            $response['data'] = $data;
        }

        if ($redirectUrl) {
            $response['redirect_url'] = $redirectUrl;
        }

        echo rssap_json_encode($response);

        exit;
    }


    /**
     * Display successfully updated notice
     */
    public function displayNotice() {
        ?>
        <div class="updated"><p><?php _e( 'Settings have been saved successfully', 'rss-autopilot' ); ?></p></div>
    <?php
    }


    /**
     * Validate settings form
     * @param $data
     * @return ValidateForm
     */
    private function getSettingsFormValidator($data)
    {
        $form = new \RSSAutopilot\ValidateForm();
        $form->setData($data);

        $form->addField(
            'yandex_translator_api_key',
            __('Yandex Translator API key', 'rss-autopilot'),
            array()
        );

        return $form;
    }
}

?>